<?php
declare(strict_types=1);

namespace Cawl\HostedCheckout\Model;

use Magento\Framework\Exception\LocalizedException;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Model\OrderFactory;
use OnlinePayments\Sdk\Domain\GetHostedCheckoutResponse;
use Cawl\PaymentCore\Api\OrderStateManagerInterface;
use Cawl\PaymentCore\Api\QuoteResourceInterface;
use Cawl\PaymentCore\Api\SessionDataManagerInterface;
use Cawl\PaymentCore\Model\Order\RejectOrderException;
use Cawl\PaymentCore\Model\OrderState\OrderState;
use Cawl\PaymentCore\Model\QuotePayment\QuotePaymentRepository;

class ReturnRequestProcessor
{
    public const SUCCESS_STATE = 'success';
    public const WAITING_STATE = 'waiting';
    public const FAIL_STATE = 'fail';

    /**
     * @var SessionDataManagerInterface
     */
    private $sessionDataManager;

    /**
     * @var QuoteResourceInterface
     */
    private $quoteResource;

    /**
     * @var OrderFactory
     */
    private $orderFactory;

    /**
     * @var OrderStateManagerInterface
     */
    private $orderStateManager;

    /**
     * @var SuccessTransactionChecker
     */
    private $successTransactionChecker;

    /**
     * @var QuotePaymentRepository
     */
    private $quotePaymentRepository;

    public function __construct(
        SessionDataManagerInterface $sessionDataManager,
        QuoteResourceInterface $quoteResource,
        OrderFactory $orderFactory,
        OrderStateManagerInterface $orderStateManager,
        SuccessTransactionChecker $successTransactionChecker,
        QuotePaymentRepository $quotePaymentRepository
    ) {
        $this->sessionDataManager = $sessionDataManager;
        $this->quoteResource = $quoteResource;
        $this->orderFactory = $orderFactory;
        $this->orderStateManager = $orderStateManager;
        $this->successTransactionChecker = $successTransactionChecker;
        $this->quotePaymentRepository = $quotePaymentRepository;
    }

    /**
     * @param string $paymentId
     * @param string $returnId
     * @return OrderState|null
     * @throws LocalizedException
     * @throws RejectOrderException
     */
    public function processRequest(string $paymentId, string $returnId): ?OrderState
    {
        if (!$paymentId || !$returnId) {
            throw new LocalizedException(__('Invalid request'));
        }

        $quote = $this->quoteResource->getQuoteByWorldlinePaymentId($paymentId);
        if (!$quote) {
            return null;
        }

        $payment = $quote->getPayment();
        $payment->setAdditionalInformation('payment_id', $paymentId);
        $quotePayment = $this->quotePaymentRepository->getByPaymentIdentifier($paymentId);
        $payment->setMethod($quotePayment->getMethod());
        $quote->setIsActive(false);
        $this->quoteResource->save($quote);

        $response = $this->successTransactionChecker->check($quote, $paymentId);

        $reservedOrderId = (string)$quote->getReservedOrderId();
        $order = $this->orderFactory->create()->loadByIncrementId($reservedOrderId);
        if (!$order->getId()) {
            return $this->processWaitingState($quote, $response);
        }

        return $this->processSuccessState($quote, $order);
    }

    private function processWaitingState(CartInterface $quote, GetHostedCheckoutResponse $response): OrderState
    {
        $reservedOrderId = (string)$quote->getReservedOrderId();
        $this->sessionDataManager->reserveOrder($reservedOrderId);
        $paymentCode = (string)$quote->getPayment()->getMethod();

        $orderState = $this->orderStateManager->create($reservedOrderId, $paymentCode, self::WAITING_STATE);

        $redirectPaymentMethodSpecificOutput = $response->getCreatedPaymentOutput()
            ->getPayment()
            ->getPaymentOutput()
            ->getRedirectPaymentMethodSpecificOutput();

        if ($redirectPaymentMethodSpecificOutput) {
            $paymentProductId = (int) $redirectPaymentMethodSpecificOutput->getPaymentProductId();
            $this->quoteResource->setPaymentIdAndSave($quote, $paymentProductId);
            $orderState->setPaymentProductId($paymentProductId);
        }

        return $orderState;
    }

    private function processSuccessState(CartInterface $quote, OrderInterface $order): OrderState
    {
        $this->sessionDataManager->setOrderData($order);

        return $this->orderStateManager->create(
            (string)$quote->getReservedOrderId(),
            (string)$quote->getPayment()->getMethod(),
            self::SUCCESS_STATE
        );
    }
}
