<?php
declare(strict_types=1);

namespace Cawl\HostedCheckout\Gateway\Response;

use Magento\Framework\Exception\LocalizedException;
use Magento\Payment\Gateway\Response\HandlerInterface;
use Magento\Sales\Model\Order\Payment;
use OnlinePayments\Sdk\Domain\DataObject;
use OnlinePayments\Sdk\Domain\CaptureResponse;
use OnlinePayments\Sdk\Domain\GetHostedCheckoutResponse;
use Cawl\PaymentCore\Api\SubjectReaderInterface;

class TransactionIdHandler implements HandlerInterface
{
    /**
     * @var SubjectReaderInterface
     */
    private $subjectReader;

    public function __construct(
        SubjectReaderInterface $subjectReader
    ) {
        $this->subjectReader = $subjectReader;
    }

    /**
     * @param array $handlingSubject
     * @param array $response
     * @return void
     * @throws LocalizedException
     */
    public function handle(array $handlingSubject, array $response): void
    {
        $paymentDO = $this->subjectReader->readPayment($handlingSubject);

        if ($paymentDO->getPayment() instanceof Payment) {
            /** @var DataObject $response */
            $response = $response['object'] ?? false;
            if (!$response) {
                return;
            }

            if ($response instanceof GetHostedCheckoutResponse) {
                $transaction = $response->getCreatedPaymentOutput()->getPayment();
            } elseif ($response instanceof CaptureResponse) {
                $transaction = $response;
            } else {
                throw new LocalizedException(__('Worldline Incorrect Response'));
            }

            $orderPayment = $paymentDO->getPayment();
            $this->setTransactionId($orderPayment, $transaction);

            $orderPayment->setIsTransactionClosed($this->shouldCloseTransaction());
            $closed = $this->shouldCloseParentTransaction($orderPayment);
            $orderPayment->setShouldCloseParentTransaction($closed);
        }
    }

    protected function setTransactionId(Payment $orderPayment, DataObject $transaction): void
    {
        $orderPayment->setTransactionId($transaction->getId());
    }

    /**
     * Whether transaction should be closed
     *
     * @return bool
     */
    protected function shouldCloseTransaction(): bool
    {
        return false;
    }

    /**
     * Whether parent transaction should be closed
     *
     * @param Payment $orderPayment
     * @return bool
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    protected function shouldCloseParentTransaction(Payment $orderPayment): bool
    {
        return false;
    }
}
